#ifndef __ECore__
#define __ECore__

/**
 * @mainpage Infinity API
 *
 * @image html logo.png
 *
 * @section licence_sec Licence
 *
 * The Infinity API is licenced to you, the end user under the terms and conditions defined in the licence document found in the 'Documentation' folder.\n
 * If you did not recieve a copy of the licence with this source code, please refer to http://www.expdigital.co.uk
 * \n
 *
 * @section ded_sec Dedications / Thanks
 * The Infinity API would not exist without the help and support of several people:\n
 * First there are our long suffering other halves - Cathy and Jo. You are both absolute stars! Thanks for putting up with all the late nights, missed dinners and for reminding us to eat and sleep!\n
 * Secondly a big thank you has to go out to all the staff at FXpansion, and a specific mention for SKot McDonald, who has helped in so many ways that its impossible to list them all. He is a king amongst men.\n
 * Thanks also go to all the guys and gals who have supported us and given us kind words of encouragment. There are too many to list all of them but a couple deserve specific mention:\n
 * Quinn Capen (API beta tester extrordinaire!)\n
 * Mully and The Betabugs crew\n
 * The KVR Developers forum\n
 *\n
 * And finally a big thanks to everyone who downloads and uses Infinity!
 *
 * @section intro_sec Introduction
 *
 * The Infinity API is a cross platform (OSX, Win32) toolkit
 *\n
 *\n
 * It is divided in to a series of libraries to enable easy maintainence and to ensure that projects can be built with the minimum amount of extra source code.\n
 * The foundation library, upon which all other libraries depend is ECore. The is the primary library and as such it provides those generic classes that are most required when developing applications.\n
 * Note that this does not include any Graphical-Interface classes. These are handled by a seperate library (EGUI).\n
 * \n
 * <B>Project structure</B>\n
 * Each library is subdivided in to a set of namespaces. Each namespace is also contained within a corresponding folder. So, for example, if a class is contained inside the 'Host' namespace it will be found in a folder named 'Host' as-well.\n
 * For example, ECore contains the following namespace / folders:
 * <UL>
 *    <LI>Basics - The highest level folder in the entire API. Defined within this namespace are high level of objects that handle class implementation issues, string handling, class managment and wrappers around the built in C++ types
 *	  <LI>Collections - Templated object containers, as well as auto pointers and template deletion handlers
 *    <LI>Exceptions - Classes for handlign and throwing exceptions
 *    <LI>Host - Classes to enable interfacing within the host in a cross platform manner. This class enables the programmer to access host OS information as well as interogation of the file systems, graphics system (also see EGUI) etc
 *    <LI>IO - File and disk i/o streams
 *    <LI>MathTools - Mathematical routines
 *    <LI>Testing - Classes for quick and easy debugging and testing of code
 *    <LI>Threading - Threads, critical sections etc
 * </UL>
 *
 * The Infinity API is based around a class management system. This is one of the most important features of the API's. A good place to start when first learning the use of the API is to take a look at the Basics namespace\n
 * This namespace features several objects that should be fully understood before any other objects are looked at. They are:
 * @see Exponent::Basics::ICountedObject
 * @see Exponent::Basics::CClass
 *
 * @section libs_sec Other Libraries in the Infinity API
 *
 * There are several other parts to the Infinity API, that is:
 * <UL>
 *		<LI>EGUI - This library provides the graphical sub system for the Infinity API and allows you to create windows, make controls, load images etc\n
 * The EGUI library is based around the idea of multiple lightweight 'controls'. Each control is a simple class that extends the IControl interface. Every control has to sit inside an IControlRoot. This interface provides a 'hook' between the actual window\n
 * that is hosting the application and the controls underneath. To group panels together you can use CControlPanel which inherits from both IControlRoot and IControl to give a control that can host other controls. Generally if you want to implement a special type of control grouping you will inherit from CControlPanel.\n
 * Controls have a generic drawing function, that is passed a CGraphics object. This is the actual graphics context that doesnt the cross platform drawing. In the EGUI framework all drawing is done using the 'double buffering' technique, meaning that there is no flicking or glitching when performing multiple drawing calls.\n
 * For most of the functionality of the framwork to function, we have implemented several types of 'listener'. A listener is a delegate object that is notified of an event having occurred, such as mouse movement, changes to the host window etc.\n
 * Each of these events has its own specific type and its own corresponding listener. This design decision gives you the knowledge that you are handling one single case inside a delegate function.
 *		<LI>EMUSIC - This library provides routines for audio and midi load and save as well as computations around musical values
 *		<LI>EVST - The EVst library provides wrappers and helper classes around the Steinberg VST2.4 (!) plug-in specification.\n
 * It expects the VST project files as supplied by Steinberg to be located in the "Third Party" folder. Simply follow the directions in the Source folder.\n
 * Unlike other Infinity API libraries it is not split in to sub namespaces. This is because of the parsity of the classes
 * </UL>
 *
 * @section pointers_sec Pointers and management in the Infinity API
 * It is worth pointing out that in the Infinity API all pointers to ICountedObjecteds are reference counted. You shouldnt ever directly delete a reference counted object\n
 * instead, use the macro FORGET_COUNTED_OBJECT. This has an important implication for the function headers:\n
 * If a function takes a pointer to a counted object you can assume that this object is being reference counted. In this case *never* pass is an address of a local variable\n
 * @code
 * // Say we have a function like this that takes a pointer to a counted object
 * void foo(CCountedObject *object)
 *
 * // If you call it with the address of a local variable then potentially a non dynamic variable will be deleted
 * CCountedObject myObject;
 * foo(&myObject);					// << NEVER, EVER, EVER do this!
 *
 * // The correct way to handle it is like this
 * CCountedObject *myObject = new CCountedObject;
 * myObject->referenced();			// << We want to remember this object for the time being
 * foo(myObject);					// << DO do this!
 * // Do something else with myObject
 * // ...
 * // ...
 * FORGET_COUNTED_OBJECT(myObject);
 * @endcode
 * \n
 * However, if a function takes the address of a variable, then you are safe to pass in a local variable:
 * @code
 * // Takes an address
 * void foo(const CCountedObject &object);
 *
 * // Then you are safe to do both
 * CCountedObject myObject;
 * foo(myObject);
 *
 * // And
 * CCountedObject *myObject = new CCountedObject;
 * foo(*myObject);
 * // Do something else with myObject
 * // ...
 * // ...
 * FORGET_COUNTED_OBJECT(myObject);
 * @endcode
 * \n
 * When storing a counted object, always make sure that you reference the new copy and dereference the old pointer. We have a handy macro for this: EXCHANGE_COUNTED_OBJECTS
 * @code
 * void foo(CCountedObject *object)
 * {
 *     EXCHANGE_COUNTED_OBJECTS(myObject, object);		// << Dereferences myObject, stores object in myObject pointer and references object
 * }
 * @endcode
 * @see FORGET_COUNTED_OBJECT
 * @see EXCHANGE_COUNTED_OBJECTS
 *
 * @section support_sec Support and Help
 *
 * Please direct all comments about the code to our developers support forum on our website:\n
 * http://support.expdigital.co.uk
 * \n
 * \n
 * We hope that you enjoy using the Infinity API\n
 * Team Exponent
 *
 * @section version_sec Version History
 * Version 1.0.1 - Maintenance release, adding several new features and fixing a couple of small bugs\n
 * <UL>
 * 	<LI>Added VST 2.4 Support
 * 	<LI>Added drawing support for Composted windows on OSX
 *	<LI>Added PNG Image support
 *	<LI>Updated CKnob::onDoubleClick to send action event
 *  <LI>Fixed several documentation bugs / typos
 *	<LI>Improved windows GDI+ startup and shutdown to only occur during graphics object startup and shutdown
 *	<LI>Improved windows menu drawing code (should be much quicker with sub menus now)
 *	<LI>Improved windows font sizing (they are now true typographic Postscript point sizes now!)
 * </UL>
 * Version 1.0.0 - Initial release
 *
 */

/**
 * @file ECore.h
 * @brief Single include for the ECore API
 *
 * @date 06/04/2006
 * @author Paul Chana
 * @version 1.0.0 Initial version
 *
 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.\n
 * Certain sections of this code may come from other sources. They are credited where applicable.\n
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 *
 * @todo Remove CBounds
 * @todo Sort out CMathTools
 *
 * $Id: ECore.h,v 1.15 2007/03/07 22:54:14 paul Exp $
 */


/**
 * @namespace Exponent
 * @brief Main Exponent namespace
 *
 * All Infinity classes should be contained within <B>both</B> the Exponent namespace and their own sub namespace. this avoids namespace collisions with other libraries.\n
 * It should exactly match the folder structure to get to the class
 */

//	===========================================================================

#include "Basics/Basics.h"
#include "Collections/Collections.h"
#include "Exceptions/Exceptions.h"
#include "Host/Host.h"
#include "IO/IO.h"
#include "MathTools/MathTools.h"
#include "Testing/Testing.h"
#include "Threading/Threading.h"

//	===========================================================================

#ifndef EXPONENT_COMPACT_BUILD
	using namespace Exponent;
#endif

//	===========================================================================

#ifndef __API_INITIALISATION__
#define __API_INITIALISATION__
	/** @cond */
	#ifdef WIN32
	/** @endcond */
		/**
	  	 * @def START_EXPONENT_API(hInstance, AppDeveloper, AppName, AppDescription, AppVersion, AppRegistry, AppCopyright)
	  	 * Call this to correctly start up a dll or other application that requires application information\n
	  	 * \a hInstance Is the application instance\n
	  	 * \a AppDeveloper A string representing the application developer (eg ExpDigital)\n
	  	 * \a AppName A string representing the name of the application  (eg GFlux)\n
	  	 * \a AppDescription A string description (in freeform text) describing the application\n
	  	 * \a AppVersion The application version as a string in form Major.Minor.Revision (eg 1.0.2)\n
	  	 * \a AppRegistry The registry location, usually spomething like Software\\Exponent\\GFlux refering to HKLM
		 * \a AppCopyright A string representing the application copyright information\n
		 * This is a <B>Win32</B> Version of the startup routine
		 * @note All of these defines have the name Exponent in them. This is for historical reasons, and they have not been changed to avoid breaking code!
	  	 */
		#define START_EXPONENT_API(hInstance, AppDeveloper, AppName, AppDescription, AppVersion, AppRegistry, AppCopyright)\
			CApplication::setApplicationInstance(hInstance);\
			CApplication::setApplicationModuleHandle(hInstance);\
			CApplication::setApplicationDeveloper(AppDeveloper);\
			CApplication::setApplicationName(AppName);\
			CApplication::setApplicationDescription(AppDescription);\
			CApplication::setApplicationVersion(AppVersion);\
			CApplication::setApplicationRegistry(AppRegistry);\
			CApplication::setApplicationCopyrightInformation(AppCopyright);\
			CApplication::setApplicationInstallPathFromApplicationHandle();\
			CFPUInformation::initialiseCPUInformation();\
			CFont::createDefaultFont();
	/** @cond */
	#else
	/** @endcond */
		/**
	  	 * @def START_EXPONENT_API(AppDeveloper, AppName, AppDescription, AppVersion, AppRegistry, AppCopyright)
	  	 * Call this to correctly start up a dll or other application that requires application information\n
	  	 * \a AppDeveloper A string representing the application developer (eg ExpDigital)\n
	  	 * \a AppName A string representing the name of the application  (eg GFlux)\n
	  	 * \a AppDescription A string description (in freeform text) describing the application\n
	  	 * \a AppVersion The application version as a string in form Major.Minor.Revision (eg 1.0.2)\n
	  	 * \a AppRegistry The registry location, usually spomething like com.exponent.gflux
		 * \a AppCopyright A string representing the application copyright information\n
		 * This is a <B>MACOSX</B> Version of the startup routine
	  	 */
		#define START_EXPONENT_API(AppDeveloper, AppName, AppDescription, AppVersion, AppRegistry, AppCopyright)\
			CApplication::setApplicationModuleHandle(AppRegistry);\
			CApplication::setApplicationDeveloper(AppDeveloper);\
			CApplication::setApplicationName(AppName);\
			CApplication::setApplicationDescription(AppDescription);\
			CApplication::setApplicationVersion(AppVersion);\
			CApplication::setApplicationRegistry(AppRegistry);\
			CApplication::setApplicationCopyrightInformation(AppCopyright);\
			CApplication::setApplicationInstallPathFromApplicationHandle();\
			CFPUInformation::initialiseCPUInformation();\
			CFont::createDefaultFont();
	/** @cond */
	#endif		// #ifdef WIN32
	/** @endcond */

	/**
	 * @def START_EXPONENT_API_CLI(AppDeveloper, AppName, AppDescription, AppVersion, AppCopyright)
	 * Class this if you want to use the Exponent API from a command line program\n
	 * \a AppDeveloper A string representing the application developer (eg ExpDigital)\n
	 * \a AppName A string representing the name of the application  (eg GFlux)\n
	 * \a AppDescription A string description (in freeform text) describing the application\n
	 * \a AppVersion The application version as a string in form Major.Minor.Revision (eg 1.0.2)\n
	 * \a AppCopyright A string representing the application copyright information\n
	 */
	#define START_EXPONENT_API_CLI(AppDeveloper, AppName, AppDescription, AppVersion, AppCopyright)\
		CApplication::setApplicationDeveloper(AppDeveloper);\
		CApplication::setApplicationName(AppName);\
		CApplication::setApplicationDescription(AppDescription);\
		CApplication::setApplicationVersion(AppVersion);\
		CApplication::setApplicationCopyrightInformation(AppCopyright);
	/**
	 * @def STOP_EXPONENT_API(ClassLogPath)
	 * Call this to correctly stop your application from using the Exponent API (destructor on mac, dll release on windows). Note that this is intended for use in graphical applications\n
	 * ie - Those using EGUI. You will recieve an error if EGUI based objects arent available\n
	 * \a ClassLogPath is a string path that is the location to save the class log files
	 * @see STOP_EXPONENT_API_NO_LOG_ECORE_ONLY
	 */
	#define STOP_EXPONENT_API(ClassLogPath)\
		CFont::destroyDefaultFont();\
		CImageManager::clearImageLibrary();\
		CClass::deleteClassManager(ClassLogPath);
	/**
	 * @def STOP_EXPONENT_API_NO_LOG
	 * Call this to correctly stop your application from using the Exponent API (destructor on mac, dll release on windows. This version odesnt wrtite the class log list. Note that this is intended for use in graphical applications\n
	 * ie - Those using EGUI. You will recieve an error if EGUI based objects arent available\n
	 * @see STOP_EXPONENT_API_NO_LOG_ECORE_ONLY
	 */
	#define STOP_EXPONENT_API_NO_LOG\
		CFont::destroyDefaultFont();\
		CImageManager::clearImageLibrary();\
		CClass::deleteClassManager(NULL);
	/**
	 * @def STOP_EXPONENT_API_ECORE_ONLY(ClassLogPath)
	 * Call this to correctly stop your application from using the Exponent API (destructor on mac, dll release on windows), if you are only using ECORE.
	 */
	#define STOP_EXPONENT_API_ECORE_ONLY(ClassLogPath) CClass::deleteClassManager(ClassLogPath);
	/**
	 * @def STOP_EXPONENT_API_NO_LOG_ECORE_ONLY
	 * Call this to correctly stop your application from using the Exponent API (destructor on mac, dll release on windows), if you are only using ECORE. This version odesnt wrtite the class log list
	 */
	#define STOP_EXPONENT_API_NO_LOG_ECORE_ONLY CClass::deleteClassManager(NULL);
#endif		// __API_INITIALISATION__

#endif	// End of ECore.h